//#define debug
#include <debug.h>
#include <makros.h>
#include <RCReceive.h>
/*
  _12Switch.ino - 1/2 Kanal RC Switch  - Version 0.3
 
 Copyright (c) 2012 Wilfried Klaas.  All right reserved.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/*
 Schaltprogramm für den Arduino. 
 Der Empfänger wird vom Pin 2 gelesen, 
 die Ausgänge sind auf Pin 0 und 1.
 Über den Pin 3/4 wird gelesen, ob der jeweilige Kanal im Impulsbetrieb oder Dauerbetrieb geschaltet werden soll.
 Impulsbetrieb: einmaliges Umlegen des Hebels schaltet ein bzw. aus
 Dauerbetrieb: das Relais ist solange geschlossen, wie der Hebel in die entsprechende Richtung gelegt wird.
 
 Progbrammablauf:
 Der Nullpunkt wird zunächst durch Interpolation der ersten 10 Werte vom Empfänger festgestellt.
 Fehlerhafte Werte, also Werte ausserhalb von 900-2100 werden ignoriert. Werden mehr als 3 fehlerhafte Werte hintereinander gefunden, 
 werden die Ausgänge auf null geschaltet. (Failsafe) 
 Die Schaltschwelle für den Normalbetrieb ist auf 250ms festgelegt.
 */
 
// Blinken pro Kanal
//#define BLINK_1;
//#define BLINK_2;

// Blinkzeit in ms;
const int BLINKTIME_1 = 1000; 
const int BLINKTIME_2 = 1000; 

// Anschlusspin Empfänger
const byte PIN_RC = 2;

// Ausgänge
const byte CAN_1 = 0;
const byte CAN_2 = 1;

// Eingänge
const byte PIN_IMPULS_1 = 3;
const byte PIN_IMPULS_2 = 4;

/* Konstante für die Erkennung des Schaltbefehls. 
 Bei Kanal 1 wäre das Nullpunkt + SWITCH_STEP bei Kanal 2 entsprechend Nullpunkt - SWITCH_STEP
 */
const int SWITCH_STEP = 64;

RCReceive rcReceiver;

// Impulseinstellung pro Kanal
boolean impuls_1;
boolean impuls_2;

// Speicher für alte Werte pro Kanal
boolean oldMustSwitch_can1 = 0;
boolean oldMustSwitch_can2 = 0;

// Zwischenspeicher
int myRcValue;
int maxSwitch;
int minSwitch;

// aktueller Schaltzustand des jeweiligen Kanals
boolean can_1_on;
boolean can_2_on;

void setup() {
  initDebug();
  // Kanäle auf Ausgang, und dann deaktivieren
  pinMode(CAN_1, OUTPUT);     
  pinMode(CAN_2, OUTPUT);     
  digitalWrite(CAN_1, LOW);
  digitalWrite(CAN_2, LOW);

  // Eingang für Impuls 1
  pinMode(PIN_IMPULS_1, INPUT_PULLUP);     

  // Eingang für Impuls 2
  pinMode(PIN_IMPULS_2, INPUT_PULLUP);     

  // Eingang für RC
  rcReceiver.attach(PIN_RC);

  maxSwitch = 0;
  minSwitch = 0;

  can_1_on = false;
  can_2_on = false;
}

void loop() {
  // Lesen der aktuellen Impuls-Einstellungen von den Kanälen
  impuls_1 = digitalRead(PIN_IMPULS_1) == 0;
  impuls_2 = digitalRead(PIN_IMPULS_2) == 0;

  // Aktuellen RC-Wert lesen
  rcReceiver.poll();

  // Fehlerfall
  if (rcReceiver.hasError()) {
    // failsafe ...
    can_1_on = false;
    can_2_on = false;
    //    digitalWrite(CAN_1, LOW);
    //    digitalWrite(CAN_2, LOW);
  } 
  else if (rcReceiver.hasNP()) {
    if (maxSwitch == 0) {
      // Schaltschwellen definieren, nur wenn noch nicht definiert
      maxSwitch = rcReceiver.getNP() + SWITCH_STEP;
      minSwitch = rcReceiver.getNP() - SWITCH_STEP;
    }
    doWork();
  }
  switch1();
  switch2();  
}

void doWork() {
  myRcValue = rcReceiver.getValue();
  // Kanal 1 auf Impuls?
  if (impuls_1) {
    doImpuls1();
  } 
  else {
    // Ausgang schalten.
    can_1_on = myRcValue > maxSwitch;
    //    digitalWrite(CAN_1, myRcValue > maxSwitch);
  }
  // Kanal 2 auf Impuls?
  if (impuls_2) {
    doImpuls2();
  } 
  else {
    // direkt schalten
    can_2_on = myRcValue < maxSwitch;
    //    digitalWrite(CAN_2, myRcValue < minSwitch);
  }
}

// Impulsschalten für Kanal 1
void doImpuls1() {
  // Änderung am Empfängerwert ?
  boolean mustSwitch = myRcValue > maxSwitch;
  if (oldMustSwitch_can1 != mustSwitch) {
    // neuen Wert schreiben, aber nur an der positiven Flanke 
    if (mustSwitch) {
      can_1_on = !can_1_on;
      //      digitalWrite(CAN_1, !digitalRead(CAN_1 ));
    }
    oldMustSwitch_can1 = mustSwitch;
  }
}

// Impulsschalten für Kanal 2
void doImpuls2() {
  // Änderung am Empfängerwert
  boolean mustSwitch = myRcValue < minSwitch;
  if (oldMustSwitch_can2 != mustSwitch) {
    // Nur Schalten bei negativer Flanke
    if (mustSwitch) {
      can_2_on = !can_2_on;
      //      digitalWrite(CAN_2, !digitalRead(CAN_2 ));
    }
    oldMustSwitch_can2 = mustSwitch;
  }
}

void switch1() {
  if (can_1_on) {
#ifdef BLINK_1
    if ((millis() % BLINKTIME_1) < (BLINKTIME_1 / 2)) {
      digitalWrite(CAN_1, HIGH);      
    } 
    else {
      digitalWrite(CAN_1, LOW);
    }
#else
    digitalWrite(CAN_1, HIGH);
#endif
  } 
  else {
    digitalWrite(CAN_1, LOW);
  }
}

void switch2() {
  if (can_2_on) {
#ifdef BLINK_2
    if ((millis() % BLINKTIME_2) < (BLINKTIME_2 / 2)) {
      digitalWrite(CAN_2, HIGH);      
    } 
    else {
      digitalWrite(CAN_2, LOW);
    }
#else
    digitalWrite(CAN_2, HIGH);
#endif
  } 
  else {
    digitalWrite(CAN_2, LOW);
  }
}

