#include <debug.h>
#include <makros.h>
#include <RCReceive.h>
/*
  _56Switch.ino - 5/6 Kanal RC Switch  - Version 0.2
 
 Copyright (c) 2013 Wilfried Klaas.  All right reserved.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/*
 Schaltprogramm für den Arduino mit 5 bzw. 6 Kanälen. 
 Der Empfänger wird vom Pin 2 gelesen, 
 die Ausgänge sind auf Pin 4,5,6,7,8,9.
 
 Progbrammablauf:
 Der Nullpunkt wird zunächst durch Interpolation der ersten 10 Werte vom Empfänger festgestellt.
 Fehlerhafte Werte, also Werte ausserhalb von 900-2100 werden ignoriert. Werden mehr als 3 fehlerhafte Werte hintereinander gefunden, 
 werden alle Ausgänge auf null geschaltet. (Failsafe) 
 Die Schaltschwelle für den Normalbetrieb ist auf 250ms festgelegt.
 Gemessen wird die Zeit, wie lange der Hebel in die eine oder andere Richtung gelegt wurde. 
 Kurz (<1s) werden Kanal 1 bzw. Kanal 2 geschaltet
 Mittel(1s>x>3s) werden die Kanäle 2und 4 geschaltet
 Lang (>3s) werden die Kanäle 5/6 geschaltet
 Statt dem Kanal 6 kann auch ein kompletter Reset aller Kanäle ausgelöst werden. 
 Dazu dient der Compilerschalter DO_RESET
 
 Jeder Kanal kann mit einer Blinkfunktion ausgestattet werden. 
 Dazu muss in dem Feld BLINKTIME für den jeweiligen Kanal ein Wert <> 0 eingestellt werden. 
 Die Zahl wird dann als Zeitspanne gewertet. (Angabe in ms) 
 Für die Zuordnung von Kanälen zu Pins ist ads Feld CAN zuständig.
 Die beiden Zeitschranken TIME_1 und TIME_2 definieren die Zeitschranken zwischen den Umschaltpunkten.
 */

// Statt Kanal 6 kann auch ein Reset aller Funktionen ausgeführt werden.
#define DO_RESET;

// Welcher Kanal soll Blinken? Blinkzeit > 0 in ms;
// HINWEIS: Die Felder gehen ab Index 0 los, d.h. Kanal 1 ist Index 0 Kanal 2 ist Index 1 ... Kanal 6 ist Index 5
const int BLINKTIME[] = {
  0, 1000, 0, 0, 0, 0}; 

const byte PIN_RC = 2;

// Pins der Ausgänge
const byte CANNEL[] = {
  4, 5, 6, 7, 8, 9};

/* Konstante für die Erkennung des Schaltbefehls. 
 Bei Kanal 1 wäre das Nullpunkt + SWITCH_STEP bei Kanal 2 entsprechend Nullpunkt - SWITCH_STEP
 */
const int SWITCH_STEP = 64;

// erste Schaltschwelle
const unsigned long TIME_1 = 1000;
// zweite Schaltschwelle
const unsigned long TIME_2 = 3000;

RCReceive rcReceiver;

// Zwischenspeicher
int myRcValue;
int maxSwitch;
int minSwitch;

// aktueller Schaltzustand des jeweiligen Kanals
boolean can_on[6];

void setup() {
  // Kanäle auf Ausgang, und dann deaktivieren
  for (int i = 0; i < 6; i++) {
    pinMode(CANNEL[i], OUTPUT);     
    digitalWrite(CANNEL[i], LOW);
    can_on[i] = false;
  }
  // Eingang für RC
  rcReceiver.attach(PIN_RC);

  maxSwitch = 0;
  minSwitch = 0;
}

void loop() {
  // Aktuellen RC-Wert lesen
  rcReceiver.poll();

  // Fehlerfall
  if (rcReceiver.hasError()) {
    // failsafe ...
    switchAllOff();
  } 
  else if (rcReceiver.hasNP()) {
    if (maxSwitch == 0) {
      // Schaltschwellen definieren, nur wenn noch nicht definiert
      maxSwitch = rcReceiver.getNP() + SWITCH_STEP;
      minSwitch = rcReceiver.getNP() - SWITCH_STEP;
    }
    doWork();
  }
  for (int i = 0; i < 6; i++) {
    doSwitch(i);
  }
}

boolean up, down = false;
unsigned long saveTime;

void doWork() {
  byte myRcValue = rcReceiver.getValue();

  if (myRcValue > maxSwitch) {
    if (!up) {
      up = true;
      down = false;
      saveTime = millis();
    }
  } 
  else if (myRcValue < minSwitch) {
    if (!down) {
      down = true;
      up = false;
      saveTime = millis();
    }
  } 
  else {
    if (up) {
      // Zurück vom vorwärts schalten
      up = false;
      saveTime = millis() - saveTime;
      if (saveTime < TIME_1) {
        can_on[0] = !can_on[0];
      } 
      else if (saveTime < TIME_2) {
        can_on[2] = !can_on[2];
      } 
      else {
        can_on[4] = !can_on[4];
      }
    } 
    else 
      if (down) {
      // Zurück vom vorwärts schalten
      down = false;
      saveTime = millis() - saveTime;
      if (saveTime < TIME_1) {
        can_on[1] = !can_on[1];
      } 
      else if (saveTime < TIME_2) {
        can_on[3] = !can_on[3];
      } 
      else {
#ifdef DO_RESET
        switchAllOff();
#else
        can_on[5] = !can_on[5];
#endif
      }
    }
  }
}

void switchAllOff() {
  for (int i = 0; i < 6; i++) {
    can_on[i] = false;
    doSwitch(i);
  }
}

void doSwitch(int can) {
  int blinktime = BLINKTIME[can];
  if (can_on[can]) {
    if (blinktime > 0) {
      if ((millis() % blinktime) < (blinktime / 2)) {
        digitalWrite(CANNEL[can], HIGH);      
      } 
      else {
        digitalWrite(CANNEL[can], LOW);
      }
    } 
    else {
      digitalWrite(CANNEL[can], HIGH);
    }
  } 
  else {
    digitalWrite(CANNEL[can], LOW);
  }
}
