#include <debug.h>
#include <makros.h>
#include <RCReceive.h>
#include <Servo.h>

/*
 Kettenfahrzeuge.ino - Programm zur Steuerung eines Kettenfahrzeuges - Version 0.1 
 
 Copyright (c) 2013 Wilfried Klaas.  All right reserved.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/*
 Der Empfänger wird vom Pin 2 (Throttle) und 3 (Steering) gelesen
 Ausgabe an 2 ESC's erfolgt am Pin 9,10
 */

// Wie stark soll der Einfluss der Steuerung auf die Motoren sein. 
// ACHTUNG: Höhere Werte ergeben weniger Effekt
const byte STEERING_IMPACT = 2;

// Pieperdefinitionen
//#define BACK_BEEPER
#define BEEPER

// Blinklicht schaltbar
// #define BACK_BLINK
// #define ALL_BLINK
// #define DRIVE_BLINK

// Hardwareanbindung für Arduino Hardware
// Empfängerkanäle
const byte PIN_RC_THR = 2; 
const byte PIN_RC_STE = 3;

// Ausgänge
const byte LED_BLINK = 7;
const byte PIN_BEEP = 8;
const byte SERVO_1 = 9; // PWM Kanal 
const byte SERVO_2 = 10; // PWM Kanal
const byte LED = 13; // LED auf dem Board

// Ab hier nitte nichts mehr ändern.
RCReceive escReceiver;
RCReceive servoReceiver;

Servo esc1, esc2;

void setup() {
  // Kanäle auf Ausgang, und dann deaktivieren
  pinMode(LED, OUTPUT);
  digitalWrite(LED, LOW);

  pinMode(LED_BLINK, OUTPUT);
  digitalWrite(LED_BLINK, LOW);

  // Eingang für RC
  escReceiver.attachInt(PIN_RC_THR);
  servoReceiver.attachInt(PIN_RC_STE);

  // Servos definieren
  esc1.attach(SERVO_1);
  esc2.attach(SERVO_2);

  esc1.write(90);
  esc2.write(90);    
}

void loop() {
  if (escReceiver.hasError() || servoReceiver.hasError()) {
    // fail safe
    esc1.write(90);
    esc2.write(90);    
  }
  else if (escReceiver.hasNP() && servoReceiver.hasNP()) {
    doWork();
  }
  else {
    int value = escReceiver.getValue();
    value = servoReceiver.getValue();
  }
}

boolean blinkOn = false;

void doWork() {
  // Werte holen
  int escValue = escReceiver.getValue();
  int servoValue = servoReceiver.getValue();

  // Nullpunkte festlegen
  escValue = escValue - escReceiver.getNP();
  servoValue = servoValue - servoReceiver.getNP();

  // Beschränken des Servoweges
  servoValue = servoValue / STEERING_IMPACT;

  // die eigentliche Berechnung
  int servo1 = escValue + servoValue;
  int servo2 = escValue - servoValue;

  // Und damit das nicht aus dem Ruder läuft, etwas begrenzen
  servo1 = constrain(servo1, -128, 128);
  servo2 = constrain(servo2, -128, 128);

#ifdef BACK_BEEPER 
  if ((servo1 < -3) || (servo2 < -3)) {
    doBeep();
  }
#endif

#ifdef BEEPER 
  if ((abs(servo1) > 3) || (abs(servo2) > 3)) {
    doBeep();
  }
#endif

#ifdef ALL_BLINK
  blinkOn = true;
  doBlink();
#endif

#ifdef BACK_BLINK
  blinkOn = (servo1 < -3) || (servo2 < -3);
  doBlink();
#endif

#ifdef DRIVE_BLINK
  blinkOn = (abs(servo1) > 3) || (abs(servo2) > 3);
  doBlink();
#endif

  // Auf den Servo umsetzen
  servo1 = map(servo1,-128, 128,0,180);
  servo2 = map(servo2,-128, 128,0,180);

  // und einstellen
  esc1.write(servo1);
  esc2.write(servo2);
}

boolean toneOn = false;

void doBeep() {
  if ((millis() % 1000) < 500) {
    if (!toneOn) {
      tone(PIN_BEEP, 600, 500);
      toneOn = true;
    }
  } 
  else {
    toneOn = false;
  }
}

void doBlink() {
  if (blinkOn) {
    if ((millis() % 1000) < 500) {
      digitalWrite(LED_BLINK, HIGH);
    } 
    else {
      digitalWrite(LED_BLINK, LOW);
    }
  } 
  else {
    digitalWrite(LED_BLINK, LOW);
  }
}
