//#define debug
#include <RCReceive.h>
#include <debug.h>
#include <makros.h>

/*
 automatische Lichtsteurung für RC Autos und Trucks für den Arduino. 
 Der Empfänger wird vom Pin 2 und 3 gelesen, 
 D2 ist das ESC
 D3 ist der Servo
 
 Die Ausgänge sind folgendermassen definiert:
 
 D0: Fahrlicht rechts
 D1: Fahrlicht links
 D4: Fernlicht
 D5: Stand/Fahrlicht rechts (PWM)
 D6: Stand/Fahrlicht links (PWM)
 D7: Blinker rechts
 D8: Blinker links
 D9: Bremslicht
 D10: Rück/Bremslicht (PWM)
 D11: Rückfahrscheinwerfer rechts
 D12: Rückfahrscheinwerfer links
 D13: schaltbares Dauerlicht (Nebellicht)
 
 Die Blinker gehen automatisch bei mehr als 50% Lenkeinschlag an und Blinken mit ca. 1Hz.
 Standlicht ist an, sobald das System einsatzbereit ist. Fahrlicht geht an, wenn der 
 Nullpunkt überschritten wird. Bei mehr als 50% Ausschlag wird das Fernlicht eingeschaltet.
 
 Wird das erste mal auf Rückwärts geschaltet, wird das Bremslicht aktiviert, 
 beim 2. Mal dann die Rückfahrscheinwerfer.
 Die PWM Kanäle (D5/D6/D10) haben mehrfache Bedeutung. Ist Standlicht aktiviert, 
 leuchten die Lampen mit ca. 20% Leistung, bei Fahrlicht bzw. Bremse jeweils mit 100%.
 
 Die beiden zusätzlichen Funktionen werden durch 2 maliges Auslösen der Lenkung in die entsprechende
 Richtung aktiviert, bzw. wieder deaktiviert.
 
 Um das Programm zu installieren muss sowohl die RCReceive Bibliothek, wie auch die 
 MCSTools Bibliothek in der Arduino IDE installiert sein. Beide bekommt als Paket auf meiner 
 I-Net Seite: http://www.wk-musik.tk
 
 Und nun wünsch ich viel Spass beim Fahren.
 
 Dipl.-Ing. Wilfried Klaas
 
 */

// generelle Programmkonstanten zum Einstellen bestimmter Eigenschaften
// Blinker zwischen links und rechts tauschen, einfach einkommentieren (also das // weg... (falls man sich doch mal verlötet hat...:-) )
//#define BLINKER_TAUSCHEN

// Zusatzfunktion Nebellicht
//#define SWITCH_NEBEL_ACTIVE

// Zusatzfunktion Warnblinker an/ausschalten
//#define SWITCH_BLINKER_ACTIVE

// Lichstärke der Rücklichter (0..255)
const byte PWM_HALF_HECK = 50;
// Lichstärke Standlicht (0..255)
const byte PWM_HALF_FRONT = 70;

// ----------------------------------------
// Hardwareanbindung für Arduino Hardware
// ----------------------------------------
// Empfängerkanäle
const byte PIN_RC_THR = 2; // das ist INT 0 aber Pin 2!!!!
const byte PIN_RC_STE = 3; // das ist INT 1 aber Pin 3!!!!

// Ausgänge
const byte L_FAHRLICHT_RE = 0;
const byte L_FAHRLICHT_LK = 1;
const byte L_FERNLICHT = 4;
const byte L_STANDLICHT_RE = 5; // PWM Kanal 
const byte L_STANDLICHT_LK = 6; // PWM Kanal 
const byte L_BLK_RE = 7;
const byte L_BLK_LK = 8;

const byte L_BREMSE = 9;
const byte L_RUECKLICHT = 10; // PWM Kanal
const byte L_RUECKFAHRLICHT_RE = 11;
const byte L_RUECKFAHRLICHT_LK = 12;
const byte L_NEBEL = 13;
const byte LED = 13;

// Definition einiger Schranken für die RC Erkennung
// Obere und untere Schranke der Nullpunkterkennung
// Werte höher als TOP und tiefer als BOTTOM werden als nicht Null erkannt
const byte NP = 128;
const byte NP_JIT = 4;
const byte TOP = NP + NP_JIT;
const byte BOTTOM = NP - NP_JIT;

// Hier die 50% Schranken
const byte JIT_50 = 64;
const byte TOP_50 = NP + JIT_50;
const byte BOTTOM_50 = NP - JIT_50;

// PWM Definitionen für halbe Leistung und volle Leistung
const byte PWM_FULL = 255;

// Anzahl der Millisekunden, die zum Schalten der Sonderfunktionen 
// zwischen den 2 Betätigungen, vergehen dürfen. 
#ifdef debug
const unsigned int SCHALTZEIT = 5000;
#else
const unsigned int SCHALTZEIT = 3000;
#endif

// ------------------------------------------------------------
// Ab hier kommen Definition für die Software selber
// Hier nur etwas ändern, wenn man sicher ist, was man tut...
// ------------------------------------------------------------

RCReceive escReceiver;
RCReceive servoReceiver;

void setup() {
  // Kanäle auf Ausgang, und dann deaktivieren
  pinMode(L_STANDLICHT_RE, OUTPUT);
  digitalWrite(L_STANDLICHT_RE, LOW);
  pinMode(L_STANDLICHT_LK, OUTPUT);
  digitalWrite(L_STANDLICHT_LK, LOW);
  pinMode(L_RUECKLICHT, OUTPUT);
  digitalWrite(L_RUECKLICHT, LOW);
  pinMode(L_FAHRLICHT_RE, OUTPUT);
  digitalWrite(L_FAHRLICHT_RE, LOW);
  pinMode(L_FAHRLICHT_LK, OUTPUT);
  digitalWrite(L_FAHRLICHT_LK, LOW);
  pinMode(L_BREMSE, OUTPUT);
  digitalWrite(L_BREMSE, LOW);

  pinMode(L_BLK_LK, OUTPUT);
  digitalWrite(L_BLK_LK, LOW);
  pinMode(L_BLK_RE, OUTPUT);
  digitalWrite(L_BLK_RE, LOW);

  pinMode(L_RUECKFAHRLICHT_RE, OUTPUT);
  digitalWrite(L_RUECKFAHRLICHT_RE, LOW);
  pinMode(L_RUECKFAHRLICHT_LK, OUTPUT);
  digitalWrite(L_RUECKFAHRLICHT_LK, LOW);

  pinMode(L_FERNLICHT, OUTPUT);
  digitalWrite(L_FERNLICHT, LOW);
  pinMode(L_NEBEL, OUTPUT);
  digitalWrite(L_NEBEL, LOW);

  pinMode(LED, OUTPUT);
  digitalWrite(LED, LOW);

  // Eingang für RC

  escReceiver.attach(PIN_RC_THR);
  servoReceiver.attach(PIN_RC_STE);

#ifndef __AVR_ATtinyX4__ 
#ifdef debug
  Serial.begin(57600); 
  Serial.flush();
  // Brauchts für den Leonardo
#ifdef __AVR_ATmega32U4__
  while(!Serial) {
  }
#endif
  Serial.println("RC Lights");
  delay(100);
#endif
#endif
}

void loop() {
  // Aktuellen RC-Wert Gas lesen
  escReceiver.poll();
  dbgOut("E:");
  dbgOut(escReceiver.getValue());

  // und gleich auch für's Steuerservo
  servoReceiver.poll();
  dbgOut(",S:");
  dbgOut(servoReceiver.getValue());
  dbgOutLn();

  // Nullpunktsbestimmung ?
  if (escReceiver.hasNP() && servoReceiver.hasNP()) {
    doHeadLights();
    doBlinker();
    #ifdef SWITCH_NEBEL_ACTIVE || SWITCH_BLINKER_ACTIVE
      doSwitches();
    #endif
  }

#ifdef debug
  // Verzögerung im DebugModus, damit man überhaupt was erkennt.
  delay(100);
#endif
}

/*
Fahr- und Rücklichter auswerten. 
 */

// mögliche Stati des Fahrlichtes 
enum HEADLIGHT { 
  STAND, DRIVE, HIGHDRIVE, BRAKE, BACK };

HEADLIGHT headLightState = STAND;
byte oldEscValue = 0;
boolean hasBrake = false;
boolean standAfterBrake = false;

void doHeadLights() {
  byte rcValue = escReceiver.getValue();
  if (between(rcValue, escReceiver.getNP() - NP_JIT, escReceiver.getNP() + NP_JIT)) {
    // Im Nullpunkt Standlicht einschalten
    headLightState = STAND;
    if (hasBrake) {
      standAfterBrake = true;
    }
  } 
  else if (rcValue > (escReceiver.getNP() + JIT_50)) {
    // Fernlicht
    headLightState = HIGHDRIVE;
    hasBrake = false;
  } 
  else if (rcValue > (escReceiver.getNP() + NP_JIT)) {
    // Fahrlicht
    headLightState = DRIVE;
    hasBrake = false;
  } 
  else {
    // Rückfahrscheinwerfer oder doch nur Bremslichter?
    if (standAfterBrake && hasBrake) {
      headLightState = BACK;
    } 
    else {
      headLightState = BRAKE;
      hasBrake = true;
      standAfterBrake = false;
    }
  }
  oldEscValue = rcValue;
  showHeadLights();
}

/*
Beleuchtung entsprechend dem Status setzen
 */
void showHeadLights() {
  dbgOut("H:");
  dbgOutLn(headLightState);
  switch (headLightState) {
  case STAND:
    analogWrite(L_STANDLICHT_RE, PWM_HALF_FRONT);
    analogWrite(L_STANDLICHT_LK, PWM_HALF_FRONT);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT_RE, 0);
    digitalWrite(L_FAHRLICHT_LK, 0);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT_RE, 0);
    digitalWrite(L_RUECKFAHRLICHT_LK, 0);
    digitalWrite(L_FERNLICHT,0);
    break;
  case DRIVE:
    analogWrite(L_STANDLICHT_RE, PWM_FULL);
    analogWrite(L_STANDLICHT_LK, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT_RE, 1);
    digitalWrite(L_FAHRLICHT_LK, 1);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT_RE, 0);
    digitalWrite(L_RUECKFAHRLICHT_LK, 0);
    digitalWrite(L_FERNLICHT,0);
    break;
  case HIGHDRIVE:
    analogWrite(L_STANDLICHT_RE, PWM_FULL);
    analogWrite(L_STANDLICHT_LK, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT_RE, 1);
    digitalWrite(L_FAHRLICHT_LK, 1);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT_RE, 0);
    digitalWrite(L_RUECKFAHRLICHT_LK, 0);
    digitalWrite(L_FERNLICHT,1);
    break;
  case BRAKE:
    analogWrite(L_STANDLICHT_RE, PWM_FULL);
    analogWrite(L_STANDLICHT_LK, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_FULL);
    digitalWrite(L_FAHRLICHT_RE, 1);
    digitalWrite(L_FAHRLICHT_LK, 1);
    digitalWrite(L_BREMSE, 1);
    digitalWrite(L_RUECKFAHRLICHT_RE, 0);
    digitalWrite(L_RUECKFAHRLICHT_LK, 0);
    digitalWrite(L_FERNLICHT,0);
    break;
  case BACK:
    analogWrite(L_STANDLICHT_RE, PWM_FULL);
    analogWrite(L_STANDLICHT_LK, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT_RE, 1);
    digitalWrite(L_FAHRLICHT_LK, 1);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT_RE, 1);
    digitalWrite(L_RUECKFAHRLICHT_LK, 1);
    digitalWrite(L_FERNLICHT,0);
    break;
  }
}

/*
Blinker auswerten.
 */
enum BLINKERSTATE { 
  RIGHT, LEFT, NONE };

BLINKERSTATE blinkerState = NONE;

void doBlinker() {
  byte rcValue = servoReceiver.getValue();

  if (between(rcValue, servoReceiver.getNP() - NP_JIT, servoReceiver.getNP() + NP_JIT)) {
    // Kein Blinker
    blinkerState = NONE;
  } 
  else {
    if (rcValue > (servoReceiver.getNP() + JIT_50)) {
      blinkerState = RIGHT;
    } 
    else if (rcValue < (servoReceiver.getNP() - JIT_50)) {
      blinkerState = LEFT;
    } 
    else {
      blinkerState = NONE;
    }      
  } 
  showBlinker();
}

/*
Blinker anzeigen.
 */
void showBlinker() {
  dbgOut("B:");
  dbgOutLn(blinkerState);
  if (blinkerState == NONE) {
    // Blinker ausschalten
    digitalWrite(L_BLK_LK, 0);
    digitalWrite(L_BLK_RE, 0);    
  } 
  else {
    #ifdef BLINKER_TAUSCHEN)
      if (blinkerState == RIGHT) {
        blinkerState = LEFT;
      }
      else {
        blinkerState = RIGHT;
      }
    #endif
    // Bestimmen, ob der Blinker an oder aus sein muss.
    unsigned long actualMillis = millis();
    byte on = 0;
    if ((actualMillis % 1000) > 500) {
      on = 0;
    } 
    else {
      on = 1;
    }
    // Und wo soll geblinkt werden?
    if (blinkerState == RIGHT) {
      digitalWrite(L_BLK_RE, on);
    } 
    else {
      digitalWrite(L_BLK_LK, on);
    }
  }
}

/*
zus. Schalt- und Blinkfunktionen auswerten
 */

// das ist die Schaltfolge, die ausgeführt werden muss
enum SWITCHSTATE {
  SW_NONE, SW_FIRST, SW_NULLPOINT, SW_SECOND};

SWITCHSTATE swState = SW_NONE;
SWITCHSTATE bkState = SW_NONE;

unsigned long lastSwCalled = 0;
unsigned long lastBkCalled = 0;
boolean nebelOn  = false;
boolean warnblinkOn = false;

void doSwitches() {
  boolean changes = false;
  byte rcValue = servoReceiver.getValue();

  #ifdef SWITCH_NEBEL_ACTIVE
    // nach Schaltzeit wird die 1. Switchbetätigung automatisch zurückgenommen
    // Schaltkanal
    if ((millis() - lastSwCalled) > SCHALTZEIT) {
      swState = SW_NONE;
    }
    if (between(rcValue, servoReceiver.getNP() - NP_JIT, servoReceiver.getNP() + NP_JIT)) {
      if (swState == SW_FIRST) {
        swState = SW_NULLPOINT;
      }
    }
    if (rcValue > (servoReceiver.getNP() + JIT_50)) {
      if (swState == SW_NONE) {
        lastSwCalled = millis();
        swState = SW_FIRST;
      } 
      else if (swState == SW_NULLPOINT) {
        swState = SW_SECOND;
        nebelOn = !nebelOn;
        changes = true;
      }
    }
  #endif
  
  // hier das ganze nochmal für den Blinkkanal
  #ifdef SWITCH_BLINKER_ACTIVE
    if ((millis() - lastBkCalled) > SCHALTZEIT) {
      bkState = SW_NONE;
    }
    if (between(rcValue, servoReceiver.getNP() - NP_JIT, servoReceiver.getNP() + NP_JIT)) {
      if (bkState == SW_FIRST) {
        bkState = SW_NULLPOINT;
      }
    }
    if (rcValue < (servoReceiver.getNP() - JIT_50)) {
      if (bkState == SW_NONE) {
        lastBkCalled = millis();
        bkState = SW_FIRST;
      } 
      else if (bkState == SW_NULLPOINT) {
        bkState = SW_SECOND;
        warnblinkOn = !warnblinkOn;
        changes = true;
      }
    }
  #endif
  if (changes || warnblinkOn) {
    showSwitches();
  }
}

void showSwitches() {
  dbgOut("S:");
  dbgOut(switchOn);
  dbgOut(",");
  dbgOutLn(blinkOn);
  // Schaltkanal
  if (nebelOn) {
    digitalWrite(L_NEBEL, 1);
  } 
  else {
    digitalWrite(L_NEBEL, 0);
  }

  // Blinkkanal
  if (warnblinkOn) {
    unsigned long actualMillis = millis();
    if ((actualMillis % 1000) > 500) {
      digitalWrite(L_BLK_LK, 0);
      digitalWrite(L_BLK_RE, 0);    
    } 
    else {
      digitalWrite(L_BLK_LK, 1);
      digitalWrite(L_BLK_RE, 1);    
    }
  } 
  else {
    digitalWrite(L_BLK_LK, 0);
    digitalWrite(L_BLK_RE, 0);    
  }
}
