//#define debug
#include <RCReceive.h>
#include <debug.h>
#include <makros.h>
/*
  rc_car_light.ino - automatische Lichtsteuerung  - Version 0.1
 
 Copyright (c) 2012 Wilfried Klaas.  All right reserved.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/*
 automatische Lichtsteurung für RC Autos und Trucks für den Arduino. 
 Der Empfänger wird vom Pin 2 und 3 gelesen, 
 D2 ist das ESC
 D3 ist der Servo
 
 Die Ausgänge sind folgendermassen definiert:
 
 D4: Fahrlicht
 D5: Fernlicht
 D6: Stand/Fahrlicht (PWM)
 D7: Blinker rechts
 D8: Blinker links
 D9: Bremslicht
 D10: Rück/Bremslicht (PWM)
 D11: Rückfahrscheinwerfer
 
 Die Blinker gehen automatisch bei mehr als 50% Lenkeinschlag an und Blinken mit ca. 1Hz.
 Standlicht ist an, sobald das System einsatzbereit ist. Fahrlicht geht an, wenn der 
 Nullpunkt überschritten wird. Bei mehr als 50% Ausschlag wird das Fernlicht eingeschaltet.
 
 Wird das erste mal auf Rückwärts geschaltet, wird das Bremslicht aktiviert, 
 beim 2. Mal dann die Rückfahrscheinwerfer.
 Die PWM Kanäle (D6/D10) haben mehrfache Bedeutung. Ist Standlicht aktiviert, 
 leuchten die Lampen mit ca. 20% Leistung, bei Fahrlicht bzw. Bremse jeweils mit 100%.
 
 Um das Programm zu installieren muss sowohl die RCReceive Bibliothek, wie auch die 
 MCSTools Bibliothek in der Arduino IDE installiert sein. Beide bekommt als Paket auf meiner 
 I-Net Seite: http://www.wk-musik.tk
 
 Und nun wünsch ich viel Spass beim Fahren.
 
 Dipl.-Ing. Wilfried Klaas
 
 */

// generelle Programmkonstanten zum Einstellen bestimmter Eigenschaften
// Blinker zwischen links und rechts tauschen, einfach einkommentieren (also das // weg... (falls man sich doch mal verlötet hat...:-) )
//#define BLINKER_TAUSCHEN

// Lichstärke der Rücklichter (0..255)
const byte PWM_HALF_HECK = 50;
// Lichstärke Standlicht (0..255)
const byte PWM_HALF_FRONT = 70;

// ----------------------------------------
// Hardwareanbindung für Arduino Hardware
// ----------------------------------------
// Empfängerkanäle
const byte PIN_RC_THR = 2; // das ist INT 0 aber Pin 2!!!!
const byte PIN_RC_STE = 3; // das ist INT 1 aber Pin 3!!!!

// Ausgänge
const byte L_FAHRLICHT = 4;
const byte L_FERNLICHT = 5;
const byte L_STANDLICHT = 6; // PWM Kanal 
const byte L_BLK_RE = 7;
const byte L_BLK_LK = 8;
const byte L_BREMSE = 9;
const byte L_RUECKLICHT = 10; // PWM Kanal
const byte L_RUECKFAHRLICHT = 11;

// Definition einiger Schranken für die RC Erkennung
// Obere und untere Schranke der Nullpunkterkennung
// Werte höher als TOP und tiefer als BOTTOM werden als nicht Null erkannt
const byte NP = 128;
const byte NP_JIT = 4;
const byte TOP = NP + NP_JIT;
const byte BOTTOM = NP - NP_JIT;

// Hier die 50% Schranken
const byte JIT_50 = 64;
const byte TOP_50 = NP + JIT_50;
const byte BOTTOM_50 = NP - JIT_50;

// PWM Definitionen für halbe Leistung und volle Leistung
const byte PWM_FULL = 255;

// ------------------------------------------------------------
// Ab hier kommen Definition für die Software selber
// Hier nur etwas ändern, wenn man sicher ist, was man tut...
// ------------------------------------------------------------

RCReceive escReceiver;
RCReceive servoReceiver;

void setup() {
  // Kanäle auf Ausgang, und dann deaktivieren
  pinMode(L_STANDLICHT, OUTPUT);
  digitalWrite(L_STANDLICHT, LOW);
  pinMode(L_RUECKLICHT, OUTPUT);
  digitalWrite(L_RUECKLICHT, LOW);
  pinMode(L_FAHRLICHT, OUTPUT);
  digitalWrite(L_FAHRLICHT, LOW);
  pinMode(L_BREMSE, OUTPUT);
  digitalWrite(L_BREMSE, LOW);

  pinMode(L_BLK_LK, OUTPUT);
  digitalWrite(L_BLK_LK, LOW);
  pinMode(L_BLK_RE, OUTPUT);
  digitalWrite(L_BLK_RE, LOW);

  pinMode(L_RUECKFAHRLICHT, OUTPUT);
  digitalWrite(L_RUECKFAHRLICHT, LOW);

  pinMode(L_FERNLICHT, OUTPUT);
  digitalWrite(L_FERNLICHT, LOW);

  // Eingang für RC
  escReceiver.attach(PIN_RC_THR);
  servoReceiver.attach(PIN_RC_STE);

  initDebug();
  dbgOutLn("RC Lights");
  delay(100);
}

void loop() {
  // Aktuellen RC-Wert Gas lesen
  escReceiver.poll();
  dbgOut("E:");
  dbgOut(escReceiver.getValue());

  // und gleich auch für's Steuerservo
  servoReceiver.poll();
  dbgOut(",S:");
  dbgOut(servoReceiver.getValue());
  dbgOutLn();

  // Nullpunktsbestimmung ?
  if (escReceiver.hasNP() && servoReceiver.hasNP()) {
    doHeadLights();
    doBlinker();
#ifdef SWITCH_NEBEL_ACTIVE || SWITCH_BLINKER_ACTIVE
    doSwitches();
#endif
  }

#ifdef debug
  // Verzögerung im DebugModus, damit man überhaupt was erkennt.
  delay(100);
#endif
}

/*
Fahr- und Rücklichter auswerten. 
 */

// mögliche Stati des Fahrlichtes 
enum HEADLIGHT { 
  STAND, DRIVE, HIGHDRIVE, BRAKE, BACK };

HEADLIGHT headLightState = STAND;
byte oldEscValue = 0;
boolean hasBrake = false;
boolean standAfterBrake = false;

void doHeadLights() {
  byte rcValue = escReceiver.getValue();
  if (between(rcValue, escReceiver.getNP() - NP_JIT, escReceiver.getNP() + NP_JIT)) {
    // Im Nullpunkt Standlicht einschalten
    headLightState = STAND;
    if (hasBrake) {
      standAfterBrake = true;
    }
  } 
  else if (rcValue > (escReceiver.getNP() + JIT_50)) {
    // Fernlicht
    headLightState = HIGHDRIVE;
    hasBrake = false;
  } 
  else if (rcValue > (escReceiver.getNP() + NP_JIT)) {
    // Fahrlicht
    headLightState = DRIVE;
    hasBrake = false;
  } 
  else {
    // Rückfahrscheinwerfer oder doch nur Bremslichter?
    if (standAfterBrake && hasBrake) {
      headLightState = BACK;
    } 
    else {
      headLightState = BRAKE;
      hasBrake = true;
      standAfterBrake = false;
    }
  }
  oldEscValue = rcValue;
  showHeadLights();
}

/*
Beleuchtung entsprechend dem Status setzen
 */
void showHeadLights() {
  dbgOut("H:");
  dbgOutLn(headLightState);
  switch (headLightState) {
  case STAND:
    analogWrite(L_STANDLICHT, PWM_HALF_FRONT);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT, 0);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT, 0);
    digitalWrite(L_FERNLICHT,0);
    break;
  case DRIVE:
    analogWrite(L_STANDLICHT, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT, 1);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT, 0);
    digitalWrite(L_FERNLICHT,0);
    break;
  case HIGHDRIVE:
    analogWrite(L_STANDLICHT, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT, 1);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT, 0);
    digitalWrite(L_FERNLICHT,1);
    break;
  case BRAKE:
    analogWrite(L_STANDLICHT, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_FULL);
    digitalWrite(L_FAHRLICHT, 1);
    digitalWrite(L_BREMSE, 1);
    digitalWrite(L_RUECKFAHRLICHT, 0);
    digitalWrite(L_FERNLICHT,0);
    break;
  case BACK:
    analogWrite(L_STANDLICHT, PWM_FULL);
    analogWrite(L_RUECKLICHT, PWM_HALF_HECK);
    digitalWrite(L_FAHRLICHT, 1);
    digitalWrite(L_BREMSE, 0);
    digitalWrite(L_RUECKFAHRLICHT, 1);
    digitalWrite(L_FERNLICHT,0);
    break;
  }
}

/*
Blinker auswerten.
 */
enum BLINKERSTATE { 
  RIGHT, LEFT, NONE };

BLINKERSTATE blinkerState = NONE;

void doBlinker() {
  byte rcValue = servoReceiver.getValue();

  if (between(rcValue, servoReceiver.getNP() - NP_JIT, servoReceiver.getNP() + NP_JIT)) {
    // Kein Blinker
    blinkerState = NONE;
  } 
  else {
    if (rcValue > (servoReceiver.getNP() + JIT_50)) {
      blinkerState = RIGHT;
    } 
    else if (rcValue < (servoReceiver.getNP() - JIT_50)) {
      blinkerState = LEFT;
    } 
    else {
      blinkerState = NONE;
    }      
  } 
  showBlinker();
}

/*
Blinker anzeigen.
 */
void showBlinker() {
  dbgOut("B:");
  dbgOutLn(blinkerState);
  if (blinkerState == NONE) {
    // Blinker ausschalten
    digitalWrite(L_BLK_LK, 0);
    digitalWrite(L_BLK_RE, 0);    
  } 
  else {
#ifdef BLINKER_TAUSCHEN)
    if (blinkerState == RIGHT) {
      blinkerState = LEFT;
    }
    else {
      blinkerState = RIGHT;
    }
#endif
    // Bestimmen, ob der Blinker an oder aus sein muss.
    unsigned long actualMillis = millis();
    byte on = 0;
    if ((actualMillis % 1000) > 500) {
      on = 0;
    } 
    else {
      on = 1;
    }
    // Und wo soll geblinkt werden?
    if (blinkerState == RIGHT) {
      digitalWrite(L_BLK_RE, on);
    } 
    else {
      digitalWrite(L_BLK_LK, on);
    }
  }
}


