//#define debug
#include <debug.h>
#include <makros.h>
#include <RCReceive.h>
#include <EEPROM.h>
#include <Servo.h>
/*
  jetSteuerungsprogramm.ino - Jetsteuerung für den Arduino  - Version 0.1
 
 Copyright (c) 2012 Wilfried Klaas.  All right reserved.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 Jetsteuerung für den Arduino. 
 Der Empfänger wird vom Pin 2 gelesen, 
 die Ausgänge sind auf Pin 9 (KlappenServo)  und 10 (ESC).
 Die Nullpunkte des Kanal wird automatisch beim Start ermittelt.
 Die Steuerung erfolgt folgendermaßen: Bei Vorwärtsfahrt wird das Motorsignal ohne 
 Änderung direkt an den Regler geschickt. Der zus. Servo bleibt in der gespeicherter Position Up stehen.
 Wird nun auf Rückwärtsfahrt gestellt, werden die Reglersignale so modifiziert, das Signale für
 Vorwärtsfahrt generiert werden. Der Servo wird an die andere Position DOWN gefahren.
 
 Im Programmiermodus, PRG beimn Einschalten gedrückt, können die beiden Positionen des Servos gespeichert werden.
 Zunächst muss die obere Position gespeichert werden, dazu die Steuerung auf den gewünschten Wert einstellen 
 und PRG drücken. Zur Quittierung blinkt die LED. Danach wird die untere Position angefahren und 
 genauso gespeichert. Der Programmiermodus wird dann automatisch verlassen.
 
 Programmablauf:
 Der Nullpunkt wird zunächst durch Interpolation der ersten 10 Werte vom Empfänger festgestellt.
 Fehlerhafte Werte, also Werte ausserhalb von 900-2100 werden ignoriert. Werden mehr als 3 fehlerhafte Werte hintereinander gefunden, 
 werden die Ausgänge auf null geschaltet. (Failsafe) 
 */
const boolean HALF_BACK_THROTTLE = false;
const byte PIN_RC = 2;

// Ausgänge
const byte SERVO = 10;
const byte ESC = 9;
const byte LED = 13;

// Eingänge
// Programmswitch
const int SW_PRG = 5;

RCReceive escReceive;

Servo myServo;
Servo myEsc;

boolean klappe; 

byte minValue;
byte maxValue;
byte minEscValue;
byte maxEscValue;

void setup() {
  myServo.attach(SERVO);
  myEsc.attach(ESC);

  // Kanäle auf Ausgang, und dann deaktivieren
  pinMode(LED, OUTPUT);
  digitalWrite(LED, LOW);

  // Eingang für Impuls 1
  pinMode(SW_PRG, INPUT_PULLUP);     

  escReceive.attachInt(PIN_RC);

  minValue = EEPROM.read(0);
  if (minValue == 0xFF) {
    minValue = 45;
  }
  maxValue = EEPROM.read(1); 
  if (maxValue == 0xFF) {
    maxValue = 135;
  }

  minEscValue = EEPROM.read(2);
  if (minEscValue == 0xFF) {
    minEscValue = 90;
  }
  maxEscValue = EEPROM.read(3); 
  if (maxEscValue == 0xFF) {
    maxEscValue = 180;
  }

  myServo.write(minValue);
  klappe = false;

  myEsc.write(minEscValue);
  if (digitalRead(SW_PRG) == 0) {
    programServo();
  }

  myBlink();
}

void loop() {
  if (escReceive.hasNP()) {
    if (!escReceive.hasError()) {
      // Wir müssen nach jeder Fahrrichtungsänderung min 1/2 Sekunde warten, damit die Klappe Zeit zum fahren hat 
      byte escValue = escReceive.getValue();
      if (between(escValue, escReceive.getNP()-3, escReceive.getNP()+3)) {
        if (klappe) {
          myServo.write(minValue);
          klappe = false;        
        }
        myEsc.write(minEscValue);
      } 
      else if (escValue > escReceive.getNP()+3) {
        // Vorwärts fahren
        escValue = map(escValue, 128,255, minEscValue, maxEscValue);
        myEsc.write(escValue);
        if (klappe) {
          myServo.write(minValue);
          klappe = false;        
        }
      } 
      else {
        // Rückwärts fahren
        if (HALF_BACK_THROTTLE) {
          escValue = map(escValue, 128, 0, minEscValue, maxEscValue / 2);
        } 
        else {
          escValue = map(escValue, 128, 0, minEscValue, maxEscValue);
        }
        myEsc.write(escValue);
        if (!klappe) {
          myServo.write(maxValue);
          klappe = true;        
        }
      }
    } 
    else {
      // Fehlerfall failsafe ...
      myServo.write(minValue);
      myEsc.write(minEscValue);
    }
  }
}

typedef enum PRG_STATE { 
  PRG_UP, PRG_DOWN, PRG_MIN, PRG_MAX};

PRG_STATE state = PRG_UP;

void programServo() {
  digitalWrite(LED, HIGH);
  waitSwitch();

  while (!escReceive.hasNP()) {
    doBlink(true);  
  }


  state = PRG_UP;
  boolean ende = false;
  while (!ende) {
    digitalWrite(LED, HIGH);      

    byte escValue = escReceive.getValue();

    switch (state) {
    case PRG_UP:
      escValue = map(escValue, 0,255, 0, 180);

      myServo.write(escValue);        
      if (digitalRead(SW_PRG) == 0) {
        digitalWrite(LED, LOW);
        EEPROM.write(1, escValue);
        maxValue = escValue;
        state = PRG_DOWN;
        myBlink();
        waitSwitch();
      }
      break;
    case PRG_DOWN:
      escValue = map(escValue, 0,255, 0, 180);

      myServo.write(escValue);        
      if (digitalRead(SW_PRG) == 0) {
        digitalWrite(LED, LOW);
        EEPROM.write(0, escValue);
        minValue = escValue;
        state = PRG_MAX;
        myBlink();
        myBlink();
        waitSwitch();
      }
      break;
    case PRG_MAX:
      escValue = map(escValue, 0,255, 0, 180);

      myEsc.write(escValue);        
      if (digitalRead(SW_PRG) == 0) {
        digitalWrite(LED, LOW);
        EEPROM.write(3, escValue);
        maxEscValue = escValue;
        state = PRG_MIN;
        myBlink();
        myBlink();
        myBlink();
        waitSwitch();
      }
      break;
    case PRG_MIN:
      escValue = map(escValue, 0,255, 0, 180);

      myEsc.write(escValue);        
      if (digitalRead(SW_PRG) == 0) {
        digitalWrite(LED, LOW);
        EEPROM.write(2, escValue);
        minEscValue = escValue;
        ende = true;
        myBlink();
        myBlink();
        myBlink();
        myBlink();
        waitSwitch();
      }
      break;
    default:
      myBlink();
      ende = true;
    }
  }
  myBlink();
  myBlink();
  myBlink();
}

void myBlink() {
  digitalWrite(LED, HIGH);
  delay(200);
  digitalWrite(LED, LOW);
  delay(200);
}

void doBlink(boolean fast) {
  byte time = fast ? 500 : 1000;
  if ((millis() % time) < (time/2)) {
    digitalWrite(LED, HIGH);      
  } 
  else {
    digitalWrite(LED, LOW);
  }
}

void waitSwitch() {
  while (digitalRead(SW_PRG) == 0) {
    delay(100);
  }
}

